"""
HB_CloneZ V1.1

Last Modified: Oct/02/2018
Works with CINEMA 4D R16.050 and up.
Copyright: Holger Biebrach, www.c4dstuff.com


Name-US: HB_CloneZ
Description-US: Clones the current selected Polygons and Moves Polygons along Z-Axis[ALT: negative Direction, SHIFT: Clone connected Polygons]

Useage:
Clones the current selected Polygons and Moves Polygons along Z-Axis by the size of the selected Polygons. Using the Alt-Modifier clones the Polygons in -Z direction. 
Using the Shift-Modifier clones all connected Polygons.

Video Tutorial:
https://youtu.be/Ki6I9sRdeUU?t=17m27s
https://youtu.be/UlY-X5O137M?t=377

ChangeLog:
Jun/11/2015 V1.0
- Release Version

Oct/02/2018 V1.1
- Python Rewrite
- Optimizations
- Combined Plus and Minus Scripts in one (Alt-Modifier)
- Shift Modifier to Clone all connected Polygons

"""
import c4d
from c4d import gui,utils

def GetSelectionSize(op):

    sel = op.GetPointS()
    points = [(i, point)
          for i, point in enumerate(op.GetAllPoints())
          if sel.IsSelected(i)]

    if not points:
      return

    selsizeminX = min(point[1].x for point in points)
    selsizemaxX = max(point[1].x for point in points)
    selsizeminY = min(point[1].y for point in points)
    selsizemaxY = max(point[1].y for point in points)
    selsizeminZ = min(point[1].z for point in points)
    selsizemaxZ = max(point[1].z for point in points)

    boundbox= c4d.Vector(selsizemaxX,selsizemaxY,selsizemaxZ)-c4d.Vector(selsizeminX,selsizeminY,selsizeminZ)
    selsize=boundbox.GetLength()
    sizeX=selsizemaxX-selsizeminX
    sizeY=selsizemaxY-selsizeminY
    sizeZ=selsizemaxZ-selsizeminZ
    
    return sizeZ

def main():
    
   
    Shift=False
    AltModifier=False
    bc = c4d.BaseContainer()    
    c4d.gui.GetInputState(c4d.BFM_INPUT_KEYBOARD,c4d.BFM_INPUT_CHANNEL,bc)
    if bc[c4d.BFM_INPUT_QUALIFIER] ==4 : AltModifier=True #ALT
    if bc[c4d.BFM_INPUT_QUALIFIER] ==1 : Shift=True #Shift
    
    doc.StartUndo()
    selobj=doc.GetActiveObject()
    if Shift: c4d.CallCommand(12557)# Select Connected
    
    mode=doc.GetMode()
    if doc.GetMode()==7: # is Polymode active? Run selction Command
        c4d.utils.SendModelingCommand(command = c4d.MCOMMAND_SELECTPOLYTOPOINT,
                                    list = [selobj],
                                    mode = c4d.MODELINGCOMMANDMODE_POLYGONSELECTION,
                                    bc = c4d.BaseContainer(),
                                    doc = doc)
    else:
        return
    
    
    PolyselectionSize=GetSelectionSize(selobj)
    
    #CLONE POLYS
    doc.AddUndo(c4d.UNDOTYPE_CHANGE,selobj)
    CloneCommand = c4d.ID_MODELING_CLONE_TOOL
    mode = c4d.MODELINGCOMMANDMODE_POLYGONSELECTION
    options = c4d.BaseContainer()
    if AltModifier: options[c4d.MDATA_CLONE_OFFSET] = -PolyselectionSize
    else: options[c4d.MDATA_CLONE_OFFSET] = PolyselectionSize
    options[c4d.MDATA_CLONE_AXIS]=2
    options[c4d.MDATA_CLONE_CNT]=1
    options[c4d.MDATA_CLONE_SCALE]=1
    options[c4d.MDATA_CLONE_LOCALSCALE]= c4d.Vector(1,1,1)
    utils.SendModelingCommand(command=CloneCommand, list=[selobj], mode=mode, bc=options, doc=doc)
    
    #Optimize
    OptCommand= c4d.MCOMMAND_OPTIMIZE
    mode = c4d.MODIFY_ALL
    options = c4d.BaseContainer()
    options[c4d.MDATA_OPTIMIZE_TOLERANCE]=0.001
    options[c4d.MDATA_OPTIMIZE_POINTS] = True
    options[c4d.MDATA_OPTIMIZE_POLYGONS] = True
    utils.SendModelingCommand(command=OptCommand, list=[op], mode=mode, bc=options, doc=doc)

    doc.EndUndo()


if __name__=='__main__':
    main()
    c4d.EventAdd() 
